#!/usr/bin/bash
# py-kms offline installer - Version sécurisée pour Ubuntu 24.04
# V 2.0 02.05.25

# Définition des dossiers
FOLDER_INSTALL='/usr/local/bin/py-kms'
FOLDER_LOG='/var/log/py-kms'
OFFLINE_PACKAGE="." # Utilise le répertoire courant

# Liste des paquets essentiels à protéger
ESSENTIAL_PACKAGES="sudo bash apt dpkg systemd coreutils"

# Vérification de la distribution
if [ -f /etc/os-release ]; then
    . /etc/os-release
    echo "Système détecté: $NAME $VERSION_ID"
    
    # Vérification spécifique pour Ubuntu 24.04
    if [[ "$NAME" == "Ubuntu" && "$VERSION_ID" == "24.04" ]]; then
        echo "ATTENTION: Détection d'Ubuntu 24.04. Mode d'installation sécurisé activé."
    fi
else
    echo "Impossible de déterminer la distribution."
fi

# Vérification des sous-dossiers nécessaires
if [ ! -d "$OFFLINE_PACKAGE/debs" ] || [ ! -d "$OFFLINE_PACKAGE/pip" ] || [ ! -d "$OFFLINE_PACKAGE/py-kms" ]; then
    echo "Erreur: Un ou plusieurs dossiers requis n'existent pas."
    echo "Vérifiez que vous avez bien les dossiers:"
    echo "- debs"
    echo "- pip"
    echo "- py-kms"
    echo "dans le répertoire courant."
    exit 1
fi

# Installation de Python et pip si nécessaires (sans supprimer d'autres paquets)
echo "Vérification de Python et pip..."
if ! command -v python3 &> /dev/null || ! command -v pip3 &> /dev/null; then
    echo "Python3 ou pip3 n'est pas installé. Installation manuelle requise."
    echo "Veuillez installer Python3 et pip3 avant de continuer:"
    echo "  sudo apt update && sudo apt install python3 python3-pip -y"
    exit 1
fi

# Installation manuelle des dépendances à partir des fichiers .deb
echo "Installation des paquets .deb individuellement depuis les fichiers locaux..."
for deb in $OFFLINE_PACKAGE/debs/*.deb; do
    PACKAGE_NAME=$(dpkg-deb -f "$deb" Package)
    echo "Installation de $PACKAGE_NAME..."
    
    # Simulation pour vérifier si l'installation supprimerait des paquets essentiels
    REMOVAL_CHECK=$(sudo dpkg --install --dry-run "$deb" 2>&1 | grep "removing" || true)
    
    # Vérifier si un paquet essentiel serait supprimé
    SAFE_TO_INSTALL=true
    for essential in $ESSENTIAL_PACKAGES; do
        if echo "$REMOVAL_CHECK" | grep -q "$essential"; then
            echo "DANGER: L'installation de $PACKAGE_NAME supprimerait $essential, opération annulée!"
            echo "Détail du conflit: $REMOVAL_CHECK"
            SAFE_TO_INSTALL=false
            break
        fi
    done
    
    # Installer uniquement si c'est sécuritaire
    if [ "$SAFE_TO_INSTALL" = true ]; then
        sudo dpkg --force-confdef --force-confold -i "$deb" || true
    else
        echo "Paquet $PACKAGE_NAME ignoré pour des raisons de sécurité."
        echo "Vous devrez peut-être installer ce paquet manuellement après vérification."
    fi
done

echo "Installation des paquets Python depuis les fichiers locaux..."
sudo pip3 install --no-index --find-links=$OFFLINE_PACKAGE/pip tzlocal pysqlite3 || true

echo "Création des dossiers d'installation et de logs..."
sudo mkdir -p $FOLDER_INSTALL
sudo mkdir -p $FOLDER_LOG

echo "Copie des fichiers py-kms depuis le package offline..."
sudo cp -r $OFFLINE_PACKAGE/py-kms/* $FOLDER_INSTALL/
sudo touch $FOLDER_LOG/server.log
sudo chown $USER $FOLDER_LOG/server.log

# Vérifier si la correction Python 3.10+ est nécessaire
echo "Vérification de la compatibilité Python..."
PYTHON_VERSION=$(python3 --version | awk '{print $2}')
MAJOR_VERSION=$(echo $PYTHON_VERSION | cut -d. -f1)
MINOR_VERSION=$(echo $PYTHON_VERSION | cut -d. -f2)

# Si Python >= 3.10, s'assurer que la correction est appliquée
if [ "$MAJOR_VERSION" -eq 3 ] && [ "$MINOR_VERSION" -ge 10 ]; then
    echo "Python $PYTHON_VERSION détecté, application de la correction..."
    if grep -q "from collections import Sequence" "$FOLDER_INSTALL/Etrigan.py"; then
        sudo sed -i 's/from collections import Sequence/from collections.abc import Sequence/' "$FOLDER_INSTALL/Etrigan.py"
        echo "Correction appliquée pour la compatibilité Python 3.10+"
    else
        echo "Le fichier semble déjà compatible avec Python moderne"
    fi
fi

# Vérification si systemd est disponible de manière sécurisée
if command -v systemctl &> /dev/null && pidof systemd &> /dev/null; then
    echo "Configuration du service systemd..."
    sudo touch /etc/systemd/system/py-kms.service
    sudo bash -c "cat > /etc/systemd/system/py-kms.service << EOF
[Unit]
Description=py-kms service
After=network.target
StartLimitIntervalSec=0
[Service]
Type=simple
Restart=always
RestartSec=1
User=$USER
ExecStart=python3 $FOLDER_INSTALL/pykms_Server.py -V INFO -F $FOLDER_LOG/server.log
[Install]
WantedBy=multi-user.target
EOF"

    echo "Activation et démarrage du service py-kms via systemd..."
    sudo systemctl daemon-reload
    sudo systemctl enable py-kms
    sudo systemctl start py-kms
    sudo systemctl status py-kms || true
else
    echo "Systemd n'est pas disponible ou n'est pas le système d'init principal."
    echo "Création d'un script de démarrage alternatif..."
    
    sudo bash -c "cat > $FOLDER_INSTALL/start-pykms.sh << EOF
#!/bin/bash
python3 $FOLDER_INSTALL/pykms_Server.py -V INFO -F $FOLDER_LOG/server.log
EOF"
    
    sudo chmod +x $FOLDER_INSTALL/start-pykms.sh
    
    echo "Pour démarrer le serveur py-kms manuellement, exécutez:"
    echo "  $FOLDER_INSTALL/start-pykms.sh &"
    echo ""
    echo "Voulez-vous démarrer le serveur maintenant? (o/n)"
    read -r response
    if [[ "$response" =~ ^([oO][uU][iI]|[oO])$ ]]; then
        echo "Démarrage du serveur py-kms..."
        nohup sudo -u $USER $FOLDER_INSTALL/start-pykms.sh > /dev/null 2>&1 &
        echo "Serveur démarré avec PID: $!"
    fi
fi

echo "Installation terminée avec succès!"
echo "IMPORTANT: Si vous avez rencontré des avertissements de sécurité,"
echo "certains paquets ont peut-être été omis pour protéger votre système."
echo "Vérifiez les messages ci-dessus pour plus de détails."
